
{-------------------------------------------------------------------------------

The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
the specific language governing rights and limitations under the License.


     The Original Code is NiceSettings.pas released at June 8th, 2003.
     The Initial Developer of the Original Code is Priyatna.
     (Website: http://www.priyatna.org Email: me@priyatna.org)
     All Rights Reserved.


Alternatively, the contents of this file may be used under the terms of the
GNU General Public License Version 2 or later (the "GPL"), in which case
the provisions of the GPL are applicable instead of those above.
If you wish to allow use of your version of this file only under the terms
of the GPL and not to allow others to use your version of this file
under the MPL, indicate your decision by deleting the provisions above and
replace them with the notice and other provisions required by the GPL.
If you do not delete the provisions above, a recipient may use your version
of this file under either the MPL or the GPL.

-------------------------------------------------------------------------------}


unit NiceSettings;

interface

uses
  Classes;

type
  TStoreTarget = (stIniFile, stRegistry);

  TNiceSettings = class(TComponent)
  private
    FStoredProps: TStrings;
    FRegKey: string;
    FIniFile: string;
    FStoreTarget: TStoreTarget;
    procedure SetStoredProps(const Value: TStrings);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure LoadSettings;
    procedure SaveSettings;
  published
    property StoredProps: TStrings read FStoredProps write SetStoredProps;
    property RegKey: string read FRegKey write FRegKey;
    property IniFile: string read FIniFile write FIniFile;
    property StoreTarget: TStoreTarget read FStoreTarget write FStoreTarget;
  end;

  procedure Register;


implementation

uses
  Dialogs, Forms, SysUtils, TypInfo, Registry, IniFiles;


procedure Register;
begin
  RegisterComponents('priyatna.org', [TNiceSettings]);
end;


{ TNiceSettings }

constructor TNiceSettings.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FStoreTarget := stIniFile;
  FIniFile := 'Settings.ini';
  FRegKey := 'Software\My Company\My Program';
  FStoredProps := TStringList.Create;
end;

destructor TNiceSettings.Destroy;
begin
  FStoredProps.Free;
  inherited Destroy;
end;

procedure TNiceSettings.SetStoredProps(const Value: TStrings);
begin
  FStoredProps.Assign(Value);
end;

procedure TNiceSettings.LoadSettings;
var
  x, i: Integer;
  s: string;
  Comp: TComponent;
  CompName, PropName, PropValue: string;
  Reg: TRegistry;
  Ini: TIniFile;
  t: TStringList;

begin

  if (FStoredProps.Count = 0)
    then Exit;

  t := TStringList.Create;

  if (FStoreTarget = stRegistry) then
  begin
    Reg := TRegistry.Create;
    Reg.OpenKey(FRegKey + '\' + Owner.Name, True);
    Reg.GetValueNames(t);
  end else
  begin
    Ini := TIniFile.Create(ExtractFilePath(Application.ExeName) + FIniFile);
    Ini.ReadSection(Owner.Name, t);
  end;

  for x := 0 to t.Count-1 do
  begin
    s := t[x];
    if (s <> '') then
    begin
      i := Pos('.', s);
      CompName := Copy(s, 1, i - 1);
      if (CompName = '')
        then Comp := Owner
        else Comp := Owner.FindComponent(CompName);
      if Assigned(Comp) then
      begin
        try
          PropName := Copy(s, i + 1, Length(s));
          if (FStoreTarget = stRegistry)
            then PropValue := Reg.ReadString(s)
            else PropValue := Ini.ReadString(Owner.Name, s, '');
          if (PropValue <> '')
            then SetPropValue(Comp, PropName, PropValue);
        except
          // Hide Exception
        end;
      end;
    end;
  end;

  if (FStoreTarget = stRegistry) then
  begin
    Reg.CloseKey;
    Reg.Free
  end else
    Ini.Free;

  t.Free;

end;

procedure TNiceSettings.SaveSettings;
var
  x, i: Integer;
  s: string;
  Comp: TComponent;
  CompName, PropName, PropValue: string;
  Reg: TRegistry;
  Ini: TIniFile;
  t: TStringList;

begin

  if (FStoredProps.Count = 0)
    then Exit;

  if (FStoreTarget = stRegistry) then
  begin
    Reg := TRegistry.Create;
    Reg.OpenKey(FRegKey + '\' + Owner.Name, True);
    t := TStringList.Create;
    Reg.GetValueNames(t);
    for x := 0 to t.Count-1
      do Reg.DeleteValue(t[x]);
    t.Free;  
  end else
  begin
    Ini := TIniFile.Create(ExtractFilePath(Application.ExeName) + FIniFile);
    Ini.EraseSection(Owner.Name);
  end;
  
  for x := 0 to FStoredProps.Count-1 do
  begin
    s := Trim(FStoredProps[x]);
    if (s <> '') then
    begin
      i := Pos('.', s);
      CompName := Copy(s, 1, i - 1);
      if (CompName = '')
        then Comp := Owner
        else Comp := Owner.FindComponent(CompName);
      if Assigned(Comp) then
      begin
        try
          if (CompName <> '')
            then CompName := CompName + '.';
          PropName := Copy(s, i + 1, Length(s));
          PropValue := GetPropValue(Comp, PropName);
          if (FStoreTarget = stRegistry)
            then Reg.WriteString(CompName + PropName, PropValue)
            else Ini.WriteString(Owner.Name, CompName + PropName, PropValue);
        except
          // Hide Exception
        end;
      end;
    end;
  end;

  if (FStoreTarget = stRegistry) then
  begin
    Reg.CloseKey;
    Reg.Free
  end else
    Ini.Free;

end;

end.
